package br.com.pavesys.services;

import br.com.pavesys.beans.*;
import br.com.pavesys.beans.VO.AutenticacaoRequestVO;
import br.com.pavesys.beans.VO.AutenticacaoResponseVO;
import br.com.pavesys.beans.VO.LogoutRequestVO;
import br.com.pavesys.beans.VO.SessionCheckRequestVO;
import br.com.pavesys.beans.dao.UsuarioSessaoDAO;
import br.com.pavesys.exceptions.BusinessException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.stereotype.Service;
import org.springframework.web.server.ResponseStatusException;

import java.time.LocalDate;
import java.util.Optional;

@Service
public class AuthenticationService {

    private Logger log = LoggerFactory.getLogger(AuthenticationService.class);

    @Autowired
    private AuthenticationManager authenticationManager;

    @Autowired
    private UsuarioSessaoDAO usuarioSessaoDAO;

    @Autowired
    private TokenService tokenService;

    @Autowired
    private SessaoService sessaoService;

    public ResponseEntity loginUser(AutenticacaoRequestVO data) {

        log.debug("Iniciando login no service para username={}", data.getUsername());

        var usernamePassword = new UsernamePasswordAuthenticationToken(data.getUsername(), data.getPassword());
        var auth = this.authenticationManager.authenticate(usernamePassword);

        log.debug("Username e password autenticados com sucesso...");

        Usuario usuario = (Usuario) auth.getPrincipal();

        if (usuario.getDatExpiracao() != null && usuario.getDatExpiracao().isBefore(LocalDate.now())) {
            usuario.setStatus(UsuarioStatus.EXPIRADO);
            log.debug("Problema ao realizar login: Usuário está EXPIRADO.");
            return ResponseEntity.badRequest().body("Usuário expirado");
        }

        if (usuario.getStatus() == UsuarioStatus.SUSPENSO) {
            log.debug("Problema ao realizar login: Usuário está SUSPENSO.");
            return ResponseEntity.badRequest().body("Usuário suspenso");
        }

        try {
            var token = tokenService.generateToken(usuario);
            var role = usuario.getRole();
            var viaSegura = usuario.getViaSegura();
            var id = usuario.getId();

            log.debug("Gerando token e captando cargo do usuário...");

            sessaoService.createSession(usuario, token);
            log.debug("Usuário de username={} e token={} logado com sucesso", data.getUsername(), token);

            return ResponseEntity.ok(new AutenticacaoResponseVO(token, role, viaSegura, id));
        } catch (BusinessException exception) {
            log.error("Erro ao logar usuário", exception);
            throw new ResponseStatusException(HttpStatus.BAD_REQUEST, exception.getMessage());
        }
    }

    public ResponseEntity logoutUser(LogoutRequestVO data) {

        Optional<UsuarioSessao> usuarioSessaoOpt = usuarioSessaoDAO.findLastSessionByToken(data.getToken());

        log.debug("Realizando logout no usuário {}...", usuarioSessaoOpt.get().getUsuario().getUsername());

        if (usuarioSessaoOpt.isEmpty()) {
            log.debug("Problema ao realizar logout: Não existem sessões para este usuário");
            return ResponseEntity.badRequest().build();
        }

        log.debug("Fechando sessão do usuário...");
        sessaoService.closeSession(data.getToken());

        log.debug("Logout realizado com sucesso para o usuário {}", usuarioSessaoOpt.get().getUsuario().getUsername());
        return ResponseEntity.ok().build();
    }

    public ResponseEntity sessionCheck(SessionCheckRequestVO data) {

        log.debug("Checando sessão do usuário...");
        Optional<UsuarioSessao> usuarioSessaoOpt = usuarioSessaoDAO.findLastSessionByToken(data.getToken());

        if (usuarioSessaoOpt.isEmpty()) {
            log.debug("Falha no session-check: Usuario nao encontrado");
            return ResponseEntity.notFound().build();
        }

        log.debug("Recuperando token da ultima sessao do usuario...");
        String lastSessionToken = usuarioSessaoOpt.get().getToken();

        if (!lastSessionToken.equals(data.getToken())) {
            log.debug("Falha no session-check: O token enviado está incorreto");
            return ResponseEntity.badRequest().build();
        }

        log.debug("A sessão do usuário {} está ativa", usuarioSessaoOpt.get().getUsuario().getUsername());
        return ResponseEntity.ok(true);
    }

}
