package br.com.pavesys.services;

import br.com.pavesys.beans.Usuario;
import br.com.pavesys.beans.UsuarioRole;
import br.com.pavesys.beans.UsuarioSessao;
import br.com.pavesys.beans.VO.*;
import br.com.pavesys.beans.ViaSegura;
import br.com.pavesys.beans.dao.UsuarioDAO;
import br.com.pavesys.beans.dao.UsuarioSessaoDAO;
import br.com.pavesys.exceptions.BusinessException;
import br.com.pavesys.rests.AdminRestServices;
import jakarta.transaction.Transactional;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.stereotype.Service;
import org.springframework.web.server.ResponseStatusException;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import static br.com.pavesys.beans.UsuarioStatus.*;
import static br.com.pavesys.beans.UsuarioStatus.ATIVO;

@Service
public class AdminServices {

    @Autowired
    private UsuarioDAO usuarioDAO;

    @Autowired
    private UsuarioSessaoDAO usuarioSessaoDAO;

    @Autowired
    private PasswordEncoder passwordEncoder;

    private static final Logger logger = LoggerFactory.getLogger(AdminRestServices.class);

    public ResponseEntity registerUser(UsuarioRegistroRequestVO data) throws ResponseStatusException {

        logger.debug("Iniciando registro para o username {}", data.getUsername());

        if (usuarioDAO.findByUsername(data.getUsername()) != null) {
            logger.debug("Tentativa de registro falhou - username já registrado: {}", data.getUsername());
            return ResponseEntity.badRequest().body("Já existe um usuário registrado com este username.");
        }

        if (usuarioDAO.findByEmail(data.getEmail()) != null) {
            logger.debug("Tentativa de registro falhou - email já registrado: {}", data.getEmail());
            return ResponseEntity.badRequest().body("Já existe um usuário registrado com este email.");
        }
        if (data.getPassword().length() < 8) {
            logger.debug("Tentativa de registro falhou - a senha possui menos de 8 caracteres");
            return ResponseEntity.badRequest().body("Sua senha deve ter no minimo 8 caracteres");
        }

        if (data.getDatExpiracao() != null && data.getDatExpiracao().isBefore(LocalDate.now())) {
            logger.debug("Tentativa de registro falhou - a data de expiração definida ({}) é menor que a data atual ({})", data.getDatExpiracao(), LocalDate.now());
            return ResponseEntity.badRequest().body("A data de expiração não pode ser uma data passada.");
        }

        try {
            String encryptedPassword = passwordEncoder.encode(data.getPassword());
            Usuario newUser = new Usuario(data.getUsername(), data.getEmail(), data.getEmpresa(), encryptedPassword, data.getDatExpiracao(), data.getViaSegura(), data.getRole());
            newUser.setStatus(ATIVO);

            logger.debug("Registrando novo usuário: username={} e role={}", data.getUsername(), data.getRole());

            this.usuarioDAO.save(newUser);

            logger.debug("Usuário salvo com sucesso no banco de dados!");

            return ResponseEntity.ok("Usuário criado com sucesso");

        } catch (BusinessException exception) {
            logger.error("Erro ao registrar usuário", exception);
            throw new ResponseStatusException(HttpStatus.BAD_REQUEST, exception.getMessage());
        }
    }

    public ResponseEntity changeUserRole(TrocaDeCargoRequestVO data) {

        logger.debug("Iniciando a alteração de cargo para o usuário de id {} e role atual: ", data.getId(), data.getRole());

        if (data.getId() == null) {
            logger.debug("Tentativa de alteração de cargo falhou: O ID do usuário registrado não pode ser nulo");
            return ResponseEntity.badRequest().body("O ID do usuário não pode ser nulo.");
        }
        if (data.getRole() == null) {
            logger.debug("Tentativa de alteração de cargo falhou: O role passado não pode ser nulo");
            return ResponseEntity.badRequest().body("É necessário passar um cargo para realizar a alteração.");
        }

        Optional<Usuario> usuarioOpt = usuarioDAO.findById(data.getId());

        if (usuarioOpt.isEmpty()) {
            logger.debug("Tentativa de alteração de cargo falhou: Usuário de id {} não foi encontrado no banco de dados", data.getId());
            return ResponseEntity.notFound().build();
        }


        try {
            UsuarioRole newRole = data.getRole();
            Usuario usuario = usuarioOpt.get();

            if (usuario.getRole() == newRole) {
                logger.debug("O usuário já possui o cargo {}", data.getRole());
                return ResponseEntity.badRequest().body("O usuário já possui este cargo!");
            }

            usuario.setRole(newRole);

            logger.debug("Alterando cargo do usuário: ID={} para role={}", data.getId(), data.getRole());

            usuarioDAO.save(usuario);

            logger.debug("Cargo do usuário de ID={} alterado com sucesso para role{}", data.getId(), data.getRole());

            return ResponseEntity.ok("Cargo do usuário de ID: " + usuario.getId() + " e USERNAME: " + usuario.getUsername() + " alterado com sucesso para " + usuario.getRole() + ".");

        } catch (BusinessException exception) {
            logger.error("Erro ao alterar cargo do usuário", exception);
            throw new ResponseStatusException(HttpStatus.BAD_REQUEST, exception.getMessage());
        }

    }

    public ResponseEntity changeViaSeguraAccess(TrocaDePermissaoViaSeguraRequestVO data) {

        logger.debug("Iniciando a alteração de cargo para o usuário de id {} e acesso ViaSegura atual: ", data.getId(), data.getViaSegura());

        if (data.getId() == null) {
            logger.debug("Tentativa de alteração de permissão de acesso falhou: O ID do usuário registrado não pode ser nulo");
            return ResponseEntity.badRequest().body("O ID do usuário não pode ser nulo.");
        }
        if (data.getViaSegura() == null) {
            logger.debug("Tentativa de alteração de permissão de acesso falhou: O acesso passado não pode ser nulo");
            return ResponseEntity.badRequest().body("É necessário passar uma permissão de acesso para realizar a alteração.");
        }

        Optional<Usuario> usuarioOpt = usuarioDAO.findById(data.getId());

        if (usuarioOpt.isEmpty()) {
            logger.debug("Tentativa de alteração de permissão de acesso falhou: Usuário de id {} não foi encontrado no banco de dados", data.getId());
            return ResponseEntity.notFound().build();
        }


        try {
            ViaSegura viaSegura = data.getViaSegura();
            Usuario usuario = usuarioOpt.get();

            if (usuario.getViaSegura() == viaSegura) {
                logger.debug("O usuário já possui o acesso do viaSegura={}", data.getViaSegura());
                return ResponseEntity.badRequest().body("O usuário já possui este acesso!");
            }

            usuario.setViaSegura(viaSegura);

            logger.debug("Alterando a permissão do usuário: ID={} para acesso={}", data.getId(), data.getViaSegura());

            usuarioDAO.save(usuario);

            logger.debug("Permissão do usuário de ID={} alterado com sucesso para acesso={}", data.getId(), data.getViaSegura());

            return ResponseEntity.ok("Acesso do usuário de ID: " + usuario.getId() + " e USERNAME: " + usuario.getUsername() + " alterado com sucesso para " + usuario.getViaSegura() + ".");

        } catch (BusinessException exception) {
            logger.error("Erro ao alterar cargo do usuário", exception);
            throw new ResponseStatusException(HttpStatus.BAD_REQUEST, exception.getMessage());
        }

    }


    public ResponseEntity changeUserPassword(TrocaDeSenhaByAdminRequestVO data) {

        logger.debug("Iniciando troca de senha para o ID={}", data.getId());

        if (data.getId() == null || data.getId().describeConstable().isEmpty()) {
            logger.debug("Tentativa de mudança de senha falhou: O id do usuário não pode ser nulo");
            return ResponseEntity.badRequest().body("O ID do usuário não pode ser nulo.");
        }
        if (data.getPassword() == null || data.getPassword().isBlank()) {
            logger.debug("Tentativa de mudança de senha falhou: A nova senha não pode ser nula");
            return ResponseEntity.badRequest().body("É necessário passar uma senha para realizar a alteração.");
        }
        if (data.getPassword().length() < 8) {
            logger.debug("Tentativa de mudança de senha falhou: A nova senha deve conter no mínimo 8 caracteres");
            return ResponseEntity.badRequest().body("Sua senha deve ter no minimo 8 caracteres");
        }

        Optional<Usuario> usuarioOpt = usuarioDAO.findById(data.getId());

        if (usuarioOpt.isEmpty()) {
            logger.debug("Tentativa de mudança de senha falhou: Usuário de ID={} não encontrado no banco de dados.", data.getId());
            return ResponseEntity.notFound().build();
        }

        try {
            logger.debug("Criptografando a senha...");

            String encryptedPassword = passwordEncoder.encode(data.getPassword());

            Usuario usuario = usuarioOpt.get();

            usuario.setPassword(encryptedPassword);

            logger.debug("Alterando senha do usuário {}...", usuarioOpt.get().getUsername());

            usuarioDAO.save(usuario);

            logger.debug("Alteração de senha realizada com sucesso para o usuário {}", usuarioOpt.get().getUsername());

            return ResponseEntity.ok("Senha alterada com sucesso.");
        } catch (BusinessException exception) {
            logger.debug("Erro ao alterar senha do usuário", exception);
            throw new ResponseStatusException(HttpStatus.BAD_REQUEST, exception.getMessage());
        }
    }

    public ResponseEntity<List<ListarUsuariosResponseVO>> listUsers() {

        logger.debug("Inicializando listagem de usuários...");

        List<Usuario> usuarios = usuarioDAO.findAll();
        List<ListarUsuariosResponseVO> usuariosResponseVOList = new ArrayList<>();

        try {
            for (Usuario usuario : usuarios) {
                Integer id = usuario.getId();
                Optional<UsuarioSessao> session = usuarioSessaoDAO.findLastLoginByUserId(id);

                LocalDateTime loginTime = session.map(s -> s.getDatHorLogin()).orElse(null);

                if (usuario.getDatExpiracao() != null && usuario.getDatExpiracao().isBefore(LocalDate.now()))
                    usuario.setStatus(EXPIRADO);

                ListarUsuariosResponseVO usuariosResponseVO = new ListarUsuariosResponseVO(usuario.getId(),
                        usuario.getUsername(),
                        usuario.getEmpresa(),
                        usuario.getEmail(),
                        usuario.getRole(),
                        usuario.getViaSegura(),
                        usuario.getDatExpiracao(),
                        loginTime,
                        usuario.getStatus()
                );
                usuariosResponseVOList.add(usuariosResponseVO);
            }

            if (usuariosResponseVOList.isEmpty()) {
                logger.debug("Nenhum usuário encontrado no sistema.");
                return ResponseEntity.notFound().build();
            }

            logger.debug("Usuários listados com sucesso!");
            return ResponseEntity.ok(usuariosResponseVOList);
        } catch (BusinessException exception) {
            logger.error("Erro ao listar usuários do sistmea", exception);
            throw new ResponseStatusException(HttpStatus.BAD_REQUEST, exception.getMessage());
        }
    }

    @Transactional
    public ResponseEntity deleteUser(Integer id) {

        logger.debug("Iniciando processo de delete para o ID={}", id);

        Optional<Usuario> usuarioOpt = usuarioDAO.findById(id);

        if (usuarioOpt.isEmpty()) {
            logger.debug("Não foi possível encontrar o ID={} no banco de dados.", id);
            return ResponseEntity.notFound().build();
        }

        try {
            logger.debug("Deletando sessões do usuário...");
            usuarioSessaoDAO.deleteSessionsByUserId(id);

            logger.debug("Deletando usuário...");
            usuarioDAO.deleteById(id);

            logger.debug("Usuário de ID={} deletado com sucesso do banco de dados juntamente com suas sessões!");

            return ResponseEntity.ok("Usuário deletado com sucesso juntamente com suas sessoes!");
        } catch (BusinessException exception) {
            logger.error("Erro ao deletar usuário", exception);
            throw new ResponseStatusException(HttpStatus.BAD_REQUEST, exception.getMessage());
        }
    }

    public ResponseEntity suspendUser(Integer id) {

        logger.debug("Iniciando procedimento e suspensão do usuário de ID={}", id);

        Usuario usuario = usuarioDAO.findById(id).orElse(null);

        if (usuario == null) {
            logger.debug("Tentativa de suspensão falhou: Não foi possível encontrar o usuário no banco de dados ou o valor é null");
            return ResponseEntity.notFound().build();
        }

        if (usuario.getStatus() == SUSPENSO) {
            logger.debug("Tentativa de suspensão falhou: O usuário de ID={} já está suspenso.", id);
            return ResponseEntity.badRequest().body("Este usuário já está suspenso");
        }

        try {
            logger.debug("Suspendendo usuário de ID={}...", id);
            usuario.setStatus(SUSPENSO);

            usuarioDAO.save(usuario);
            logger.debug("Usuário de ID={} suspenso com sucesso!", id);

            return ResponseEntity.ok("Usuário suspenso com sucesso");
        } catch (BusinessException exception) {
            logger.error("Erro ao suspender usuário", exception);
            throw new ResponseStatusException(HttpStatus.BAD_REQUEST, exception.getMessage());
        }
    }

    public ResponseEntity activateUser(Integer id) {

        logger.debug("Iniciando processo de reativação para o ID={}", id);

        Usuario usuario = usuarioDAO.findById(id).orElse(null);

        if (usuario == null) {
            logger.debug("Tentativa de reativação falhou: não foi possível encontrar o usuário no banco de dados ou o valor é null");
            return ResponseEntity.notFound().build();
        }

        if (usuario.getStatus() == ATIVO) {
            logger.debug("Tentativa de reativação falhou: o usuário de ID={} já se encontra como ATIVO", id);
            return ResponseEntity.badRequest().body("Este usuário já está ativo.");
        }

        try {
            logger.debug("Reativando usuário de ID={}", id);
            usuario.setStatus(ATIVO);

            usuarioDAO.save(usuario);
            logger.debug("Usuário de ID={} reativado com sucesso!", id);

            return ResponseEntity.ok("Usuário reativado com sucesso");
        } catch (BusinessException exception) {
            logger.error("Erro ao reativar usuário", exception);
            throw new ResponseStatusException(HttpStatus.BAD_REQUEST, exception.getMessage());
        }
    }

    public ResponseEntity renewUserExpiration(Integer id, RenewExpirationRequestVO data) {

        logger.debug("Inicializando renovação de data de expiração do usuário...");

        Usuario usuario = usuarioDAO.findById(id).orElse(null);

        if (usuario == null) {
            logger.debug("Tentativa de renovação falhou: não foi possível encontrar o usuário no banco de dados ou o valor é null");
            return ResponseEntity.notFound().build();
        }

        if (data.getExpDate().isBefore(LocalDate.now())) {
            logger.debug("Tentativa de renovação falhou: a data que você passou é anterior a de hoje", id);
            return ResponseEntity.badRequest().body("Por favor, envie uma data válida.");
        }

        try {
            logger.debug("Renovando data de expiração do usuário de ID={}", id);
            usuario.setDatExpiracao(data.getExpDate());

            usuarioDAO.save(usuario);
            logger.debug("Usuário de ID={} com data de expiração renovada para: {} com sucesso!", id, usuario.getDatExpiracao());

            return ResponseEntity.ok("Usuário com data de expiração renovada com sucesso");
        } catch (BusinessException exception) {
            logger.error("Erro ao renovar usuário", exception);
            throw new ResponseStatusException(HttpStatus.BAD_REQUEST, exception.getMessage());
        }

    }

}
