package br.com.pavesys.mail;

import br.com.pavesys.beans.EsqueciMinhaSenha;
import br.com.pavesys.beans.TokenEsqueciMinhaSenhaStatus;
import br.com.pavesys.beans.Usuario;
import br.com.pavesys.beans.VO.TokenTrocaDeSenhaRequestVO;
import br.com.pavesys.beans.VO.TrocaDeSenhaByUserRequestVO;
import br.com.pavesys.beans.VO.TrocaDeSenhaTokenRequestVO;
import br.com.pavesys.beans.dao.EsqueciMinhaSenhaDAO;
import br.com.pavesys.beans.dao.UsuarioDAO;
import jakarta.transaction.Transactional;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.ResponseEntity;
import org.springframework.mail.SimpleMailMessage;
import org.springframework.mail.javamail.JavaMailSender;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.Random;

@Service
public class EmailService {

    private final Logger log = LoggerFactory.getLogger(EmailService.class);

    @Autowired
    private JavaMailSender emailSender;

    @Autowired
    EsqueciMinhaSenhaDAO esqueciMinhaSenhaDAO;

    @Autowired
    UsuarioDAO usuarioDAO;

    @Autowired
    private PasswordEncoder passwordEncoder;

    @Value("${spring.mail.username}")
    private String emailSenderUsername;

    public void sendSimpleMessage(String to, String subject, String text) {

        SimpleMailMessage message = new SimpleMailMessage();

        message.setFrom(emailSenderUsername);
        message.setTo(to);
        message.setSubject(subject);
        message.setText(text);
        emailSender.send(message);

    }

    public ResponseEntity changeUserPassword(TrocaDeSenhaByUserRequestVO data) {

        log.debug("Inicializando troca de senha do usuário");
        EsqueciMinhaSenha esqueciMinhaSenha = esqueciMinhaSenhaDAO.findByTokenNovaSenha(data.getToken());

        if (esqueciMinhaSenha == null) {
            log.debug("Não foi possível achar este token em nosso sistema.");
            return ResponseEntity.notFound().build();
        }

        if (esqueciMinhaSenha.getStatus() != TokenEsqueciMinhaSenhaStatus.ATIVO) {
            log.debug("Token já utilizado anteriormente.");
            return ResponseEntity.badRequest().body("Token já utilizado anteriormente");
        }

        if (esqueciMinhaSenha.getDatHorExpiracaoToken().isBefore(LocalDateTime.now())) {
            log.debug("Token já expirado.");
            return ResponseEntity.badRequest().body("Token já expirado.");
        }

        if (data.getPassword().length() < 8) {
            log.debug("Tentativa de mudança de senha falhou: A nova senha deve conter no mínimo 8 caracteres");
            return ResponseEntity.badRequest().body("Sua senha deve ter no minimo 8 caracteres");
        }

        Usuario usuario = esqueciMinhaSenha.getUsuario();

        log.debug("Criptografando nova senha...");
        String encryptedNewPassword = passwordEncoder.encode(data.getPassword());

        log.debug("Setando nova senha...");
        usuario.setPassword(encryptedNewPassword);

        usuarioDAO.save(usuario);
        log.debug("Nova senha definida com sucesso!");
        esqueciMinhaSenha.setStatus(TokenEsqueciMinhaSenhaStatus.USADO);
        esqueciMinhaSenhaDAO.save(esqueciMinhaSenha);
        log.debug("Status do token alterado para: UTILIZADO - impossibilitando-o de ser usado novamente");

        return ResponseEntity.ok("Senha alterada com sucesso!");
    }

    @Transactional
    public ResponseEntity createChangePasswordToken(TrocaDeSenhaTokenRequestVO data) {

        try {
            log.debug("Procurando usuário via email passado no rest...");
            Usuario usuario = usuarioDAO.findByEmail(data.getEmail());

            if (usuario == null) {
                log.debug("Usuário com o email {} não encontrado no sistema!", data.getEmail());
                return ResponseEntity.notFound().build();
            }

            log.debug("Deletando token antigo caso exista...");
            esqueciMinhaSenhaDAO.deleteByUserId(usuario.getId());

            log.debug("Gerando token...");
            LocalDateTime datHorExpiracaoToken = LocalDateTime.now().plusMinutes(30);
            String token = givenUsingJava8_whenGeneratingRandomAlphanumericString_thenCorrect();

            esqueciMinhaSenhaDAO.save(new EsqueciMinhaSenha(usuario, datHorExpiracaoToken, token, TokenEsqueciMinhaSenhaStatus.ATIVO));
            log.debug("Salvo no banco de dados: token de redefinição de senha, status do token e identificador do usuário");

            log.debug("Construindo o e-mail para envio do token...");
            String text = """
                    Olá, %s!
                    
                    Recebemos sua solicitação para redefinir sua senha. Aqui está o código para redefinição:
                    
                    Seu código de redefinição: %s
                    
                    Este código é válido por 30 minutos.
                    
                    Se você não solicitou esta redefinição, por favor, desconsidere esta mensagem.
                    
                    Para redefinir sua senha, use o código acima no nosso sistema :)
                    """.formatted(usuario.getUsername(), token);

            log.debug("Enviando email...");
            sendSimpleMessage(data.getEmail(), "Redefinição de senha", text);

            log.debug("E-mail encaminhado com sucesso!");
            return ResponseEntity.ok().build();
            } catch (RuntimeException msg) {
                log.error(msg.getMessage());
                return ResponseEntity.badRequest().body(msg.getMessage());
            }
    }

    public ResponseEntity validateChangePasswordToken(TokenTrocaDeSenhaRequestVO data) {

        log.debug("Inicializando validação do token de redefinição de senha...");
        EsqueciMinhaSenha esqueciMinhaSenha = esqueciMinhaSenhaDAO.findByTokenNovaSenha(data.getToken());

        if (esqueciMinhaSenha == null) {
            log.debug("Não foi possível achar este token em nosso sistema.");
            return ResponseEntity.notFound().build();
        }

        if (esqueciMinhaSenha.getStatus() != TokenEsqueciMinhaSenhaStatus.ATIVO) {
            log.debug("Token já utilizado anteriormente.");
           return ResponseEntity.badRequest().body("Token já utilizado anteriormente");
        }

        if (esqueciMinhaSenha.getDatHorExpiracaoToken().isBefore(LocalDateTime.now())) {
            log.debug("Token já expirado.");
            return ResponseEntity.badRequest().body("Token já expirado.");
        }

        log.debug("Token validado com sucesso!");
        return ResponseEntity.ok().build();
    }

    public String givenUsingJava8_whenGeneratingRandomAlphanumericString_thenCorrect() {

        int leftLimit = 48; // numeral '0'
        int rightLimit = 122; // letter 'z'
        int targetStringLength = 6;
        Random random = new Random();

        String generatedString = random.ints(leftLimit, rightLimit + 1)
                .filter(i -> (i <= 57 || i >= 65) && (i <= 90 || i >= 97))
                .limit(targetStringLength)
                .collect(StringBuilder::new, StringBuilder::appendCodePoint, StringBuilder::append)
                .toString();

        return generatedString;
    }


}
