﻿using Avalonia.Media;
using Newtonsoft.Json;
using Serilog;
using System;
using System.Collections.Generic;
using System.IO;

namespace pavesys_iRAP.Helper
{
    public class AppConfig
    {
        public string ActiveProfile { get; set; }
        public Dictionary<string, DatabaseProfileConfig> DbProfiles { get; set; }
        public GlobalConfig GlobalConfig { get; set; }
    }

    public class DatabaseProfileConfig
    {

        public string DatabaseName { get; set; }
        public string DatabaseUser { get; set; }
        public string DatabasePassword { get; set; }

    }

    public class GlobalConfig
    {
        // Configurações Gerais
        public string Language { get; set; }

        public int InputErrorLimit { get; set; }

        public string StretchType { get; set; }

        public string ExportPath { get; set; }

        public bool ShowTooltips { get; set; }

        public bool TwoScreenMode { get; set; }

    }

    public static class ConfigManager
    {
        private static string ConfigFilePath = Path.Combine(Environment.GetFolderPath(Environment.SpecialFolder.CommonApplicationData), "Pavesys", "iRap", "config.json");

        public static AppConfig LoadConfig()
        {
            if (File.Exists(ConfigFilePath))
            {
                var json = File.ReadAllText(ConfigFilePath);
                return JsonConvert.DeserializeObject<AppConfig>(json);
            }
            else
            {
                var defaultConfig = new AppConfig
                {
                    ActiveProfile = "Default",
                    DbProfiles = new Dictionary<string, DatabaseProfileConfig>
                    {
                        { "Default", new DatabaseProfileConfig { DatabaseName = "irapNovo", DatabaseUser = "sa", DatabasePassword = "mssql" } }
                    },
                    GlobalConfig = new GlobalConfig
                    {
                        Language = "en-US",
                        InputErrorLimit = 100,
                        StretchType = "Fill",
                        ExportPath = Path.Combine(Environment.GetFolderPath(Environment.SpecialFolder.CommonApplicationData), "Pavesys", "iRap", "Export"),
                        ShowTooltips = true,
                        TwoScreenMode = false
                    }
                };
                SaveConfig(defaultConfig);
                return defaultConfig;
            }
        }

        public static void SaveConfig(AppConfig config)
        {
            var json = JsonConvert.SerializeObject(config, Formatting.Indented);
            File.WriteAllText(ConfigFilePath, json);
        }

        public static void DeleteProfile(string profileName)
        {
            var config = LoadConfig();
            if (config.DbProfiles.ContainsKey(profileName))
            {
                config.DbProfiles.Remove(profileName);
                SaveConfig(config);
                Log.Debug("Profile '{ProfileName}' deleted successfully", profileName);
            }
            else
            {
                Log.Error("Profile '{ProfileName}' does not exist", profileName);
            }
        }
        public static DatabaseProfileConfig getActiveDbProfile()
        {
            var config = LoadConfig();
            DatabaseProfileConfig dbActiveProfile;
            bool success = config.DbProfiles.TryGetValue(config.ActiveProfile, out dbActiveProfile);
            if (success)
            {
                //Log.Debug("DatabaseProfile - '{ActiveProfile}' Loaded Successfully", config.ActiveProfile);
                return dbActiveProfile;
            }
            //TODO
            //If there's no active profile return an error
            else
                return dbActiveProfile; //Both branches are doing the same...

        }
        public static string GetLanguage()
        {
            return LoadConfig().GlobalConfig.Language;
        }

        public static int GetInputErrorLimit()
        {
            return LoadConfig().GlobalConfig.InputErrorLimit;
        }

        public static Stretch GetStretchType()
        {
            var StretchType = LoadConfig().GlobalConfig.StretchType;
            if (StretchType == "Fill")
            {
                return Stretch.Fill;
            }
            else if (StretchType == "UniformToFill")
            {
                return Stretch.UniformToFill;
            }
            else if (StretchType == "Uniform")
            {
                return Stretch.Uniform;
            }
            else
                return Stretch.None;

        }
        public static bool GetTwoScreenMode() { return LoadConfig().GlobalConfig.TwoScreenMode; }

        public static string GetExportPath()
        {
            return LoadConfig().GlobalConfig.ExportPath;
        }

        public static bool GetShowTooltips() { return LoadConfig().GlobalConfig.ShowTooltips; }

        public static void SetInputErrorLimit(int errorLimit)
        {
            var config = LoadConfig();
            config.GlobalConfig.InputErrorLimit = errorLimit;
            SaveConfig(config);
        }
        public static void SetLanguage(string language)
        {
            var config = LoadConfig();
            config.GlobalConfig.Language = language;
            SaveConfig(config);
        }

        public static void SetStretchType(string StretchType)
        {
            var config = LoadConfig();
            config.GlobalConfig.StretchType = StretchType;
            SaveConfig(config);
        }

        public static void SetExportPath(string ExportPath) 
        {
            var config = LoadConfig();
            config.GlobalConfig.ExportPath = ExportPath; 
            SaveConfig(config); 
        }

        public static void SetShowTooltips(bool showTooltips) 
        {
            var config = LoadConfig();
            config.GlobalConfig.ShowTooltips = showTooltips;
            SaveConfig(config);
        }

        public static void SetTwoScreenMode(bool twoScreenMode)
        {
            var config = LoadConfig();
            config.GlobalConfig.TwoScreenMode = twoScreenMode;
            SaveConfig(config); 
        }


    }

}
